#!/bin/bash

CONFI="$HOME/.archi/archi.conf"

if [ ! -f "$CONFI" ]; then
    echo "Fatal error: no config file ($CONFI) found!"
    exit 1
fi

arlog() {
    VALSAY="[`date '+%Y-%m-%d %H:%M:%S'`] $1"
    echo "$VALSAY" >> "$LOGFIL"
    echo "$VALSAY"
}

source "$CONFI"
mkdir -p "$HOME/.archi/logs"

LOGFIL="$HOME/.archi/logs/arh_`date +%Y-week-%V`.log"

arlog "Archize 1.00 started!"

LOCK="/tmp/archi.lock"
if [ -e "$LOCK" ]; then
    arlog "Fatal error: program already running!"
    exit 1
fi

touch "$LOCK"
trap "rm -f '$LOCK'" EXIT

arlog "Backing up started!"

if [ ! -d "$LDIR" ]; then
    arlog "Making local directory: $LDIR"
    mkdir -p "$LDIR"
fi

if [ ! -d "$TEMP" ]; then
    arlog "Making temp directory: $TEMP"
    mkdir -p "$TEMP"
fi

arlog "Pinging remote host: $RHOST"
ping -c 1 -W 2 "${RHOST#*@}" > /dev/null 2>&1
if [ $? -ne 0 ]; then
    arlog "Warn: Ping failed (dead server?). Trying SSH..."
fi

ssh -o BatchMode=yes -o ConnectTimeout=5 -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -i "$KEYFILE" "$RHOST" "exit" > /dev/null 2>&1
if [ $? -ne 0 ]; then
    arlog "Fatal error: Failed to connect to the remote host!"
    exit 1
fi
RAVAL=true

arlog "Calculating disk space..."
REQ=0
for SRC in "${DIRS[@]}"; do
    if [ -e "$SRC" ]; then
        SIZE=$(du -s "$SRC" | awk '{print $1}')
        REQ=$((REQ + SIZE))
    fi
done

AVAL=$(df "$LDIR" | awk 'NR==2 {print $4}')

arlog "Space required: ${REQ}k, Space available: ${AVAL}k"

if [ "$REQ" -gt "$AVAL" ]; then
    arlog "Fatal error: Not enough space!"
    exit 1
fi

if [ "$BKPERIOD" == "BYMONTH" ]; then
    PTAG="month_$(date +%Y-%m)"
elif [ "$BKPERIOD" == "BYWEEK" ]; then
    PTAG="week_$(date +%Y-week-%V)"
else
    PTAG="day_$(date +%Y-%m-%d)"
fi

OLD="$LDIR/old"
CTG="$LDIR/$PTAG"
REPOS="$CTG/repos"
ARCHI="$CTG/archive"

mkdir -p "$OLD"
mkdir -p "$REPOS"
mkdir -p "$ARCHI"

for SRC in "${DIRS[@]}"; do
    if [ ! -e "$SRC" ]; then
        arlog "Warn: Folder $SRC does not exist. Skipped."
        continue
    fi

    BASENAME=$(basename "$SRC")
    
    if [ -d "$SRC/.git" ]; then
        DREPO="$REPOS/$BASENAME.git"
        if [ -d "$DREPO" ]; then
            arlog "Fetching git repo $BASENAME..."
            git -C "$DREPO" fetch --all >> "$LOGFIL" 2>&1
        else
            arlog "Cloning git repo $BASENAME..."
            git clone --mirror "$SRC" "$DREPO" >> "$LOGFIL" 2>&1
        fi
    else
        arlog "RSYNCing directory $SRC..."
        RSOPTS="-av --delete --no-specials --no-devices"
        rsync $RSOPTS "$SRC" "$ARCHI/" >> "$LOGFIL" 2>&1
    fi
done

arlog "Checking if archives needs to be made..."

for DIR in "$LDIR"/*; do
    [ -d "$DIR" ] || continue
    DINAME=$(basename "$DIR")
    
    if [ "$DINAME" != "$PTAG" ] && [[ "$DINAME" =~ ^(month|week|day)_ ]]; then
        arlog "Period $DINAME is end. Archivizing."
        
        TAR="$OLD/$DINAME.tar.gz"
        
        arlog "Compressing $DINAME to $TAR..."
        tar -czf "$TAR" -C "$LDIR" "$DINAME" >> "$LOGFIL" 2>&1
        
        if [ $? -eq 0 ]; then
            arlog "Compressed succesfully."
            
            if [ "$RAVAL" = true ]; then
                arlog "Sending $DINAME.tar.gz to $RHOST..."
                
                ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -i "$KEYFILE" "$RHOST" "mkdir -p $RDIR/" >> "$LOGFIL" 2>&1
                scp -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -i "$KEYFILE" "$TAR" "$RHOST:$RDIR/" >> "$LOGFIL" 2>&1
                
                if [ $? -eq 0 ]; then
                    arlog "Sent succesfully. Cleaning temp files..."
                    rm -rf "$DIR"
                    rm -rf "$DINAME"
                else
                    arlog "Warn: Failed to send archive."
                fi
            else
                arlog "Warn: Remote host is offline."
            fi
        else
            arlog "Warn: failed to compress $DINAME"
        fi
    fi
done

arlog "Archize made it's work!"
